; $Id: read_ascii.pro,v 1.34 2000/07/11 19:18:58 kschultz Exp $
;
; Copyright (c) 1996-2000, Research Systems, Inc.  All rights reserved.
;       Unauthorized reproduction prohibited.
;+
; NAME:
;   READ_ASCII
;
; PURPOSE:
;   Read data from an ASCII file into IDL.
;
; CATEGORY:
;   Input/Output.
;
; CALLING SEQUENCE:
;   data = READ_ASCII(file)
;
; INPUTS:
;   file          - Name of file to read.
;
; INPUT KEYWORD PARAMETERS:
;   record_start      - 1st sequential "record" (see DESCRIPTION) to read.
;               Default = 0 (the first record of the file).
;   num_records   - Number of records to read.
;               Default = 0 = Read up to and including the last record.
;
;   template      - ASCII file template (e.g., generated by function
;               ASCII_TEMPLATE) describing attributes of the file
;               to read.  Specific attributes contained in the
;               template may be overridden by keywords below.
;               Default = (see the keywords below).
;
;   data_start    - Number of lines of header to skip.
;               Default (if no template) = 0L.
;   delimiter     - Character that delimits fields.
;               Default (if no template) = '' = use fields(*).loc.
;   missing_value     - Value to replace any missing/invalid data.
;               Default (if no template) = !VALUES.F_NAN.
;   comment_symbol    - String identifying comments
;               (from comment_symbol to the next end-of-line).
;               Default (if no template) = '' = no comments.
;
;    [Note:  The 'fields' keyword has not been implemented yet.]
;   fields        - Descriptions of the data fields, formatted as
;                           an array of structures containing the tags:
;                              name  = name of the field (string)
;                              type  = type of field as returned by SIZE (long)
;                              loc   = offset from the beginning of line to
;                                      the start of the field (long)
;                              group = sequential group the field is in (int)
;               Default (if no template) =
;                              {name:'field', type:4L, loc:0L, group:0}.
;
;   verbose       - If set, print runtime messages.
;               Default = Do not print them.
;
; OUTPUT KEYWORD PARAMETERS:
;   header        - The header read (string array of length
;               data_start).  If no header, empty string returned.
;
;   count         - The number of records read.
;
; OUTPUTS:
;   The function returns an anonymous structure, where each field in
;   the structure is a "field" of the data read (see DESCRIPTION).
;   If no records are read, 0 is returned.
;
; COMMON BLOCKS:
;   None.
;
; SIDE EFFECTS:
;   None.
;
; RESTRICTIONS:
;   See DESCRIPTION.
;
; DESCRIPTION:
;   ASCII files handled by this routine consist of an optional header
;   of a fixed number of lines, followed by columnar data.  Files may
;   also contain comments, which exist between a user-specified comment
;   string and the corresponding end-of-line.
;
;   One or more rows of data constitute a "record."  Each data element
;   within a record is considered to be in a different column, or "field."
;   Adjacent fields may be "grouped" into multi-column fields.
;   The data in one field must be of, or promotable to, a single
;   type (e.g., FLOAT).
;
; EXAMPLES:
;   ; Using default file attributes.
;   data = READ_ASCII(file)
;
;   ; Setting specific file attributes.
;   data = READ_ASCII(file, DATA_START=10)
;
;   ; Using a template to define file attributes.
;   data = READ_ASCII(file, TEMPLATE=template)
;
;   ; Using a template to define file attributes,
;   ; and overriding some of those attributes.
;   data = READ_ASCII(file, TEMPLATE=template, DATA_START=10)
;
;   ; Using the ASCII_TEMPLATE GUI to generate a template in place.
;   data = READ_ASCII(file, TEMPLATE=ASCII_TEMPLATE(file))
;
;    [Note:  The 'fields' keyword has not been implemented yet.]
;   ; An example defining fields by hand.
;   fields = REPLICATE({name:'', type:0L, loc:0L, group:0}, 2, 3)
;   num = N_ELEMENTS(fields)
;   fields(*).name  = 'field' + STRTRIM(STRING(INDGEN(num) + 1), 2)
;   fields(*).type  = REPLICATE(4L, num)
;   fields(*).loc   = [0L,10L, 0L,15L, 0L,12L]
;   fields(*).group = INDGEN(num)
;   data = READ_ASCII(file, FIELDS=fields)
;
;    [Note:  The 'fields' keyword has not been implemented yet.]
;   ; Another example defining fields by hand.
;   void = {sMyStructName, name:'', type:0L, loc:0L, group:0}
;   fields = [ [ {sMyStructName, 'frog', (SIZE(''))(1),  0L, 0},   $
;                {sMyStructName, 'bird', (SIZE(0 ))(1), 15L, 1} ], $
;              [ {sMyStructName, 'fish', (SIZE(0.))(1),  0L, 2},   $
;                {sMyStructName, 'bear', (SIZE(0D))(1), 15L, 3} ], $
;              [ {sMyStructName, 'boar', (SIZE(0B))(1),  0L, 4},   $
;                {sMyStructName, 'nerd', (SIZE(OL))(1), 15L, 5} ]  ]
;   data = READ_ASCII(file, FIELDS=fields)
;
; DEVELOPMENT NOTES:
;
;   - See ???,xxx in the code.
;
;   - Error check input 'delimiter' to be a string (not a byte).
;
;   - Implement the 'fields' keyword.
;
; MODIFICATION HISTORY:
;   AL & RPM, 8/96 - Written.
;   PCS, 3/99 - Deploy STRTOK and other new commands.  Gain some speed.
;-

;------------------------------------------------------------------------------
; Purpose:  Cast token (a string) to the desired type and return
;        result.  If the token can't be cast to the desired type,
;        return missing_value.
;
function ra_cast, token, type, missing_value

  COMPILE_OPT hidden, strictarr

  case 1 of
    type eq 7: $ ; String
      return, token
    token eq '': $
      return, missing_value
    else: begin
      on_ioerror, cast_failed
      return, fix(token, type=type, /print)
cast_failed:
      message, /reset
      return, missing_value
      end
  endcase

end

; -----------------------------------------------------------------------------
;
;  Purpose:  Given a start and end location into a string, parse out the
;        appropriate numerical or string value. In the case of a blank
;        string or any error, substitute in the missing value.
;
function ra_parse_value, line, type, sptr, len, $
  missing_value=missing_value

  COMPILE_OPT hidden, strictarr

  ;  Grab the sub string and remove leading/trailing white space.
  ;
  sub_str = strtrim(strmid(line, sptr, len), 2)

  return, ra_cast(sub_str, type, missing_value)

end         ; ra_parse_value

; -----------------------------------------------------------------------------
;
;  Purpose:  Parse out values from a line of text which are separated by
;        a given delimiter.
;
pro ra_parse_d_values, line, types, p_vals, rec_count, $
  delimit, missing_value, whitespace_delimited

  COMPILE_OPT hidden, strictarr

  if whitespace_delimited then $
    toks = strtok(line, /extract) $
  else $
    toks = strtrim(strtok(line, delimit, /preserve_null, /extract), 2)

  on_ioerror, cast_failed

  n_toks = n_elements(toks)
  for i=0,n_elements(types)-1 do begin
    if (types[i] gt 0) then begin ; (0 == skip field.)
      case 1 of
        i ge n_toks: $
          (*p_vals[i])[rec_count] = missing_value
        types[i] eq 7: $ ; String
          (*p_vals[i])[rec_count] = toks[i]
        toks[i] eq '': $
          (*p_vals[i])[rec_count] = missing_value
        else: begin
          (*p_vals[i])[rec_count] = fix(toks[i], type=types[i], /print)
          goto, cast_completed
cast_failed:
          message, /reset
          (*p_vals[i])[rec_count] = missing_value
cast_completed:
        end
      endcase
    endif
  endfor
end         ; ra_parse_d_values

; -----------------------------------------------------------------------------
;
;  Purpose:  Read in the next n lines of text (skipping blank lines and
;        commented lines signified by template.commentSymbol at start;
;        also throw away comment portions of regular lines).
;
pro ra_get_next_record, template, unit, lines, end_reached=end_reached
  ;
  COMPILE_OPT hidden, strictarr

  catch, error_status
  if (error_status ne 0) then begin
    end_reached = 1B
    message, /reset
    return
  endif
  ;
  line = ''
  count = 0
  end_reached = 0B

  ;  Checking for comments...
  ;
  if (template.commentSymbol ne '') then begin
    while (count lt n_elements(lines)) do begin
      readf, unit, line
      pos = strpos(line, template.commentSymbol, 0)
      if (strtrim(line,2) ne '' and pos[0] ne 0) then begin
        if (pos[0] eq -1) then lines[count] = line $
        else                   lines[count] = strmid(line,0,pos[0])
        count = count + 1
      endif
    endwhile

  ;  NOT checking for comments...
  ;
  endif else begin
    while (count lt n_elements(lines)) do begin
      readf, unit, line
      if (strtrim(line,2) ne '') then begin
        lines[count] = line
        count = count + 1
      endif
    endwhile
  endelse

end         ; ra_get_next_record

; -----------------------------------------------------------------------------
;
;  Purpose:  Given a template structure, open an ASCII file and parse out the
;        numerical and string values based upon the parameters of the
;        given template.
;
;       (a) white space separates fields lined up in columns
;       (b) a delimiter character separates fields
;
;     Note:  When skipping to the start of the data, blank lines ARE included
;        as lines to skip, but once you get to the data, subsequent blank
;        lines (as well as comment lines) are ignored.
;
;        Function returns an array of pointers to the data read;
;        if no data read, 0 is returned.
;
function ra_read_from_templ, $
    name, $     ; IN: name of ASCII file to read
    template, $     ; IN: ASCII file template
    start_record, $ ; IN: first record to read
    records_to_read, $  ; IN: number of records to read
    doVerbose, $    ; IN: 1B = print runtime messages
    num_fields_read, $  ; OUT: number of fields successfully read
    fieldNames, $       ; OUT: associated name of each field read
    rec_count, $    ; OUT: number of records successfully read
    num_blocks, $    ; OUT: number of blocks of data
    header=header   ; OUT: (opt) header read

  COMPILE_OPT hidden, strictarr

  ;  Set default numbers.
  ;
  num_fields_read = 0
  rec_count = 0l
  num_blocks = 0L

  ;  Catch errors.
  catch, error_status
  if (error_status ne 0) then begin
    print,'Unexpected Error: ' + !ERROR_STATE.msg
    rec_count = 0l
    return, 0
  endif

  ;  Open the file.
  ;
  openr, unit, name, /get_lun

  ;  Set various parameters.
  ;
  blk_size = 1000  ; each block holds this many records
  blk_count = 500  ; number of blocks we can have
  blk_grow  = 500
  current_block = 0L
  lines_per_record = n_elements(template.fieldCount)
  num_fields = template.fieldCount
  tot_num_fields = total(template.fieldCount)
  types = template.fieldTypes
  locs = template.fieldLocations

  ;  Define an array of variables for each field.
  ;
  p_vals = ptrarr(tot_num_fields, blk_count)
  for i=0, tot_num_fields-1 do $
    if (types[i] gt 0) then $
      p_vals[i, current_block] = ptr_new(make_array(blk_size, type=types[i]))

  ;  Read the header and skip to the start of the data.
  ;
  dataStart = template.dataStart
  if (dataStart gt 0) then begin
    if (doVerbose) then $
      print, 'Reading header of ' + strtrim(string(dataStart), 2) + $
        ' lines ...', format='(A/)'
    header = strarr(dataStart)
    readf, unit, header
  endif else $
    header = ''

  ;  Skip to the start of requested data.
  ;
  lines = strarr(lines_per_record)
  if ((doVerbose) and (start_record gt 0)) then $
    print, 'Skipping ' + strtrim(string(start_record), 2) + ' records ...', $
      format='(A/)'
  for i = 0L, start_record-1 do $
    ra_get_next_record, template, unit, lines

  end_reached = 0B

  ; ------------------------------------
  ; nice columned data...
  ; ------------------------------------
  ;
  if (template.delimiter eq 0B) then begin
      while (((rec_count lt records_to_read) or (records_to_read eq 0)) and $
             (not end_reached)) do begin

        ;  Read a record.
        ;
        ra_get_next_record, template, unit, lines, end_reached=end_reached
        if (not end_reached) then begin

          ;xxx
          if (doVerbose) then $
            print, 'Processing sequential record ' + $
              strtrim(string(rec_count+1), 2) + ' ...'

          ;  For each line in the record...
          ;
          anchor = 0
          for i=0,lines_per_record-1 do begin

            ;  ...parse and store values.
            ;
            for j=0, num_fields[i]-1 do begin
              if (types[anchor+j] gt 0) then begin
                if (j eq num_fields[i]-1) then $
                  len = strlen(lines[i]) - locs[anchor+j] $
                else $
                  len = locs[anchor+j+1] - locs[anchor+j]
                (*p_vals[anchor+j,current_block])[rec_count-current_block*blk_size] = $
                  ra_parse_value( $
                  lines[i], $
                  types[anchor+j], $
                  locs[anchor+j], $
                  len, $
                  missing=template.missingValue $
                  )
              endif
            endfor

            anchor = anchor + num_fields[i]
          endfor

          rec_count = rec_count + 1L

          ; If block is now full,
          ; Allocate and point to a new block
          ;
          if (rec_count mod blk_size eq 0) then begin
            current_block = current_block + 1
            if (current_block eq blk_count) then begin
                p_vals = [[p_vals], [ptrarr(tot_num_fields, blk_grow)]]
                blk_count = blk_count + blk_grow
            endif
            for i=0, tot_num_fields-1 do $
                if (types[i] gt 0) then $
                p_vals[i, current_block] = ptr_new(make_array(blk_size, type=types[i]))
          endif
        endif
      endwhile
  ; ------------------------------------
  ; data separated by a delimiter...
  ; ------------------------------------
  ;
  endif else begin
    if template.delimiter eq 32b then begin
      delim_str = string([32b, 9b])
      whitespace_delimited = 1b
    end else begin
      delim_str = string(template.delimiter)
      whitespace_delimited = 0b
    endelse

    while (((rec_count lt records_to_read) or (records_to_read eq 0)) and $
           (not end_reached)) do begin

      ;  Read the next record.
      ;
      ra_get_next_record, template, unit, lines, end_reached=end_reached
      if (not end_reached) then begin
        anchor = 0

        ;xxx
        if (doVerbose) then $
          print, 'Processing sequential record ' + $
            strtrim(string(rec_count+1), 2) + ' ...'

        ;  Loop for each line in a record, parsing values.
        ;
        for i=0, lines_per_record-1 do begin
          ra_parse_d_values, $
            lines[i], $
            types[ anchor:anchor+num_fields[i]-1], $
            p_vals[anchor:anchor+num_fields[i]-1, current_block], $
            rec_count-current_block*blk_size, $
            delim_str, $
            template.missingValue, $
            whitespace_delimited
          anchor = anchor + num_fields[i]
        endfor

        rec_count = rec_count + 1L

        ; If block is now full,
        ; Allocate and point to a new block
        ;
        if (rec_count mod blk_size eq 0) then begin
            current_block = current_block + 1
            if (current_block eq blk_count) then begin
                p_vals = [[p_vals], [ptrarr(tot_num_fields, blk_grow)]]
                blk_count = blk_count + blk_grow
            endif
            for i=0, tot_num_fields-1 do $
                if (types[i] gt 0) then $
                p_vals[i, current_block] = ptr_new(make_array(blk_size, type=types[i]))
        endif
      endif
    endwhile
  endelse
  ; ------------------------------------
  free_lun, unit

  if (doVerbose) then $
    print, 'Total records read:  ' + strtrim(string(rec_count), 2), $
      format='(A/)'

  ;  If records were read ...
  ;
  if (rec_count gt 0) then begin

    ;  Set the output arrays to exactly the correct size.
    ;
    for i=0, tot_num_fields-1 do begin
      if (p_vals[i,current_block] ne ptr_new()) then begin
        if (rec_count gt current_block*blk_size) then begin
          *p_vals[i,current_block] = $
            (*p_vals[i,current_block])[0:rec_count-current_block*blk_size-1]
        endif else begin ; block is allocated, but empty
        	ptr_free, p_vals[i,current_block]
        endelse
      endif
    endfor
    if (rec_count gt current_block*blk_size) then begin
      num_blocks = current_block + 1
    endif else begin
      num_blocks = current_block
    endelse

    ;  Check the groups array and arrange the output pointers into
    ;  (potentially) groups of 2-D arrays.
    ;
    groups = template.fieldGroups

    ;  Don't include any groups which are skipped fields.
    ;
    ptr = where(types eq 0, numSkip)
    for i=0, numSkip-1 do groups[ptr[i]] = max(groups) + 1

    ;  Concatenate 1-D arrays into multi arrays based upon groupings.
    ;
    uptr = uniq(groups, sort(groups))
    if (n_elements(uptr) lt n_elements(groups)) then begin
      for i=0, n_elements(uptr)-1 do begin
          for b=0, num_blocks-1 do begin
            lptr = where(groups eq groups[uptr[i]], lcount)
            if (lcount gt 1) then begin
              p_new = p_vals[lptr[0],b]
              for j=1,lcount-1 do begin
                *p_new = [[temporary(*p_new)],[temporary(*p_vals[lptr[j],b])]]
                ptr_free, p_vals[lptr[j],b]
                p_vals[lptr[j],b] = ptr_new()
              endfor
              *p_new = transpose(temporary(*p_new))
            endif
          endfor
      endfor
    endif

    ;  Return the pointers that contain data, if any.
    ;  and the associated fieldNames for these pointers
    ;
    ptr = where(p_vals[*,0] ne ptr_new(), num_fields_read)

    if (num_fields_read gt 0) then begin ; data successfully read
      fieldNames = template.fieldNames[ptr]
      return, p_vals[ptr,*]
    endif else begin                           ; no data read
      rec_count = 0l
      return, 0
    endelse

  endif else $                           ; no data read
    return, 0

end         ; ra_read_from_templ

; -----------------------------------------------------------------------------
;
;  Purpose:  Return 1B if the template is valid, else 0B.
;
function ra_valid_template, $
  template, $       ; IN: template to check
  message           ; OUT: error message if the template is not valid

  COMPILE_OPT hidden, strictarr

  message = ''

  ;  Make sure it's a structure.
  ;
  sz = size(template)
  if (sz[sz[0]+1] ne 8) then begin
    message = 'Template is not a structure.'
    RETURN, 0B
  endif

  ;  Get tag names and make sure version field is present.
  ;
  tagNamesFound = TAG_NAMES(template)
  void = WHERE(tagNamesFound eq 'VERSION', count)
  if (count ne 1) then begin
    message = 'Version field missing from template.'
    RETURN, 0B
  endif

  ;  Do checking based on version.
  ;
  case (template.version) of

    1.0: begin

      ;  Set the names of the required tags (version alread checked).
      ;
      tagNamesRequired = STRUPCASE([ $
        'dataStart', 'delimiter', 'missingValue', 'commentSymbol', $
        'fieldCount', 'fieldTypes', 'fieldNames', 'fieldLocations', $
        'fieldGroups'])

      ;  Check that all of the required tags are present.
      ;
      for seqTag = 0, N_ELEMENTS(tagNamesRequired)-1 do begin
        tag = tagNamesRequired[seqTag]
        void = WHERE(tagNamesFound eq tag, count)
        if (count ne 1) then begin
          message = tag + ' field missing from template.'
          RETURN, 0B
        endif
      endfor

    end

    else: begin
      message = 'The only recognized template version is 1.0 (float).'
      RETURN, 0B
    end
  endcase

  ;  Return that the template is valid.
  ;
  RETURN, 1B

end         ; ra_valid_template

; -----------------------------------------------------------------------------
;
;  Purpose:  Convert to string and remove extra white space.
;
function ra_stringit, value

  COMPILE_OPT hidden, strictarr

  result = STRTRIM( STRCOMPRESS( STRING(value) ), 2 )

  num = N_ELEMENTS(result)

  if (num le 1) then RETURN, result

  ;  If two or more values, concatenate them.
  ;
  delim = ' '
  ret = result[0]
  for i = 1, num-1 do $
    ret = ret + delim + result[i]

  RETURN, ret

end         ; ra_stringit

; -----------------------------------------------------------------------------
;
;  Purpose:  Guess at the number of columns in an ASCII file.
;
function ra_guess_columns, fname, dataStart, commentSymbol, delimiter

  COMPILE_OPT hidden, strictarr

  on_error, 2 ; Return to caller on error.

  catch, err_stat
  if err_stat ne 0 then begin
    catch, /cancel
    if n_elements(lun) gt 0 then begin
      close, lun
      free_lun, lun
    end
    message, !error_state.msg
  endif

  get_lun, lun
  openr, lun, fname

  if dataStart gt 0 then begin
    header = strarr(dataStart)
    readf, lun, header
  end

  line = ''
  ra_get_next_record, $
    {commentSymbol: commentSymbol}, $
    lun, $
    line, $
    end_reached=end_reached

  if end_reached then $
    message, 'No columns found.'

  if delimiter eq ' ' then $
    positions = strtok(line) $
  else $
    positions = strtok(line, delimiter, /preserve_null)

  close, lun
  free_lun, lun
  return, n_elements(positions)
end

; -----------------------------------------------------------------------------
;
;  Purpose: Check that the input filename is a string, exists, and appears
;           to be ASCII.
;
function ra_check_file, fname

  COMPILE_OPT hidden, strictarr

  catch, error_status
  if (error_status ne 0) then begin
    if (n_elements(unit) gt 0) then free_lun, unit
    return, -3 ; unexpected error reading from file
  endif
  ;
  info = size(fname)
  if (info[info[0]+1] ne 7) then return, -1 ; filename isn't a string
  ;
  openr, unit, fname, error=error, /get_lun
  if (error eq 0) then begin
    finfo = fstat(unit)
    ; set non-ascii values in lookup table
    ;
    lut = bytarr(256) + 1b
    lut[7:13]   = 0b
    lut[32:127] = 0b
    data = bytarr(32767<finfo.size, /nozero)
    readu, unit, data
    carriage_return = (total(data eq 10b) gt 0 or total(data eq 13b) gt 0)
    if (carriage_return eq 0) then begin
    ; looks like a binary file
    ;
      free_lun, unit
      return, -4
    endif
    non_printable   = (total(lut[data]) gt 0)
    if (non_printable) then begin
    ; looks like a binary file
    ;
      free_lun, unit
      return, -4
    endif

    free_lun, unit
  endif else $
    return, -2 ; unable to open file
end

; -----------------------------------------------------------------------------
;
;  Purpose:  The main routine.
;
function read_ascii, $
    file, $             ; IN:
    RECORD_START=recordStart, $     ; IN: (opt)
    NUM_RECORDS=numRecords, $       ; IN: (opt)
    TEMPLATE=template, $        ; IN: (opt)
    DATA_START=dataStart, $     ; IN: (opt)
    DELIMITER=delimiter, $      ; IN: (opt)
    MISSING_VALUE=missingValue, $   ; IN: (opt)
    COMMENT_SYMBOL=commentSymbol, $ ; IN: (opt)
;    FIELDS=fields, $           ; IN: (opt)    [not implemented]
    VERBOSE=verbose, $          ; IN: (opt)
    HEADER=header, $            ; OUT: (opt)
    COUNT=count             ; OUT: (opt)

    COMPILE_OPT strictarr
  ;xxx
  ;later add a VERSION kw ?

  ;  Set to return to caller on error.
  ;
  ON_ERROR, 2


  ;  Set some defaults.
  ;
  count = 0
  currentVersion = 1.0
  doVerbose = KEYWORD_SET(verbose)

  ;  If no file specified, use DIALOG_PICKFILE
  ;
  if (n_elements(file) eq 0) then begin
    file = DIALOG_PICKFILE(/MUST_EXIST)
    if (file eq '') then RETURN, 0
  endif

  ; check that the file is readable and appears to be ASCII
  ;
  ret = ra_check_file(file)
  case ret of
    -1: MESSAGE, 'File name must be a string.'
    -2: MESSAGE, 'File "' + file + '" not found.'
    -3: MESSAGE, 'Error Reading from file "' + file + '"
    -4: MESSAGE, 'File "' + file + '" is not an ASCII file.'
    else:
  endcase

  ;  Set which records to read.
  ;
  if (N_ELEMENTS(recordStart) ne 0) then recordStartUse = recordStart $
                                    else recordStartUse = 0
  if (N_ELEMENTS(numRecords) ne 0)  then numRecordsUse = numRecords $
                                    else numRecordsUse = 0
  if (N_ELEMENTS(template) gt 0) then begin
    if (not ra_valid_template(template, message)) then $
      MESSAGE, message

    versionUse      = template.version
    dataStartUse    = template.dataStart
    delimiterUse    = STRING(template.delimiter)
    missingValueUse = template.missingValue
    commentSymbolUse    = template.commentSymbol
  endif else begin
    versionUse      = currentVersion
    dataStartUse    = 0L
    delimiterUse    = ' '
    missingValueUse = !values.f_nan
    commentSymbolUse    = ''
  endelse

  if n_elements(dataStart) ne 0 then $
    dataStartUse = dataStart
  if n_elements(delimiter) ne 0 then $
    delimiterUse = delimiter
  if n_elements(missingValue) ne 0 then $
    missingValueUse = missingValue
  if n_elements(commentSymbol) ne 0 then $
    commentSymbolUse = commentSymbol

  if n_elements(dataStartUse) gt 1 then $
    message, 'DATA_START must be a scalar.'
  if n_elements(byte(delimiterUse)) gt 1 then $ ; Might want to remove this
    message, 'DELIMITER must be one character.' ; restriction in the future.

  ;  (For back version compatibility, we do not throw an error
  ;  here if n_elements(missingValueUse) gt 1).
  ;

  ;  The READ_ASCII that shipped with IDL 5.2.1 returns 0 when
  ;  an array of comment symbols is specified.  Set the error_state
  ;  in this case, but, for back version compatibility, continue
  ;  and reproduce the "return 0" behavior here.
  ;
  if n_elements(commentSymbolUse) gt 1 then begin
    message, $
      'Multiple comment symbols are not currently supported.', $
      /noprint, $
      /continue
    return, 0
  endif

  if n_elements(template) gt 0 then begin
    fieldCountUse   = template.fieldCount
    fieldTypesUse   = template.fieldTypes
    fieldNamesUse   = template.fieldNames
    fieldLocationsUse   = template.fieldLocations
    fieldGroupsUse  = template.fieldGroups
  endif else begin
    fieldCountUse = n_elements(fieldTypes) $
      > n_elements(fieldNames) $
      > n_elements(fieldLocations) $
      > n_elements(fieldGroups)
    if fieldCountUse le 0 then $
      fieldCountUse = ra_guess_columns( $
        file, $
        dataStartUse, $
        commentSymbolUse, $
        delimiterUse $
        )

    fieldTypesUse   = REPLICATE(4L, fieldCountUse)
    digits_str = strtrim(string(strlen(strtrim(string(fieldCountUse),2))),2)
    fstr = '(i' + digits_str + '.' + digits_str + ')'
    fieldNamesUse   = 'field' + STRING(INDGEN(fieldCountUse)+1, format=fstr)
    fieldLocationsUse   = LONARR(fieldCountUse)
    fieldGroupsUse  = INTARR(fieldCountUse)
  endelse

  if n_elements(fieldTypes) ne 0 then $
    fieldTypesUse = fieldTypes
  if n_elements(fieldNames) ne 0 then $
    fieldNamesUse = fieldNames
  if n_elements(fieldLocations) ne 0 then $
    fieldLocationsUse = fieldLocations
  if n_elements(fieldGroups) ne 0 then $
    fieldGroupsUse = fieldGroups

  ;  Error check the field data.
  ;
  lengths = [ $
    N_ELEMENTS(fieldTypesUse), $
    N_ELEMENTS(fieldNamesUse), $
    N_ELEMENTS(fieldLocationsUse), $
    N_ELEMENTS(fieldGroupsUse) $
    ]
  if (TOTAL(ABS(lengths - SHIFT(lengths, 1))) ne 0) then $
    MESSAGE, 'Field data (types/names/locs/groups) not the same length.'

  ;  Set the template to use.
  ;
  templateUse = { $
    version: versionUse, $
    dataStart: dataStartUse, $
    delimiter: BYTE(delimiterUse), $
    missingValue: missingValueUse, $
    commentSymbol: commentSymbolUse, $
    fieldCount: fieldCountUse, $
    fieldTypes: fieldTypesUse, $
    fieldNames: fieldNamesUse, $
    fieldLocations: fieldLocationsUse, $
    fieldGroups: fieldGroupsUse $
    }

  ;  Print verbose information.
  ;
  if (doVerbose) then begin
    PRINT, 'Using the following file attributes ...', FORMAT='(/A)'
    PRINT, '        Data Start:  ' + STRTRIM(STRING(dataStartUse), 2)
    PRINT, '         Delimiter:  ' + $
                             STRTRIM(STRING(FIX(BYTE(delimiterUse))), 2) + 'B'
    PRINT, '     Missing Value:  ' + STRTRIM(STRING(missingValueUse), 2)
    PRINT, '    Comment Symbol:  ' + commentSymbolUse
    PRINT, '      Field Counts:  ' + ra_stringit(fieldCountUse)
    PRINT, '      Field Types :  ' + ra_stringit(fieldTypesUse)
    PRINT, '      Field Names :  ' + ra_stringit(fieldNamesUse)
    PRINT, '      Field Locs  :  ' + ra_stringit(fieldLocationsUse)
    PRINT, '      Field Groups:  ' + ra_stringit(fieldGroupsUse)
    PRINT, '  Template Version:  ' + STRTRIM(STRING(versionUse), 2)
    PRINT
  endif

  ;  Try to read the file.
  ;
  pData = ra_read_from_templ(file, templateUse, recordStartUse, $
    numRecordsUse, doVerbose, numFieldsRead, FieldNames, count, num_blocks, header=header)

  ;  Return zero if no records read.
  ;
  if (count eq 0) then RETURN, 0

  ; Concatenate the blocks into fields.
  ;
  xData = ptrarr(numFieldsRead)
  for f=0L, numFieldsRead-1 do begin
    type = SIZE(*pData[f,0], /TYPE)
    dims = SIZE(*pData[f,0], /DIMENSIONS)
    n_dims = SIZE(*pData[f,0], /N_DIMENSIONS)
    dims[n_dims-1] = count
    xData[f] = ptr_new(make_array(DIMENSION=dims, TYPE=type))
    start=0L
    for b=0L, num_blocks-1 do begin
      sz = SIZE(*pData[f,b],/N_ELEMENTS)
      stop = start + sz - 1
      (*xData[f])[start:stop] = *pData[f,b]
      ptr_free, pData[f,b]
      start = stop + 1
    endfor
  endfor

  ;  Put the fields into a structure.
  ;  For a small number of fields,
  ;  use EXECUTE to build the struct all at once, instead of looping
  ;  one time for each field.  This saves a lot of copying when there
  ;  are a lot of records.
  ;
  ;  If the number of fields is larger, we cannot use EXECUTE since we run
  ;  the risk of running out of code space in the compiler, so then use
  ;  create_struct recursively.
  ;
  data = create_struct(strcompress(FieldNames[0],/rem), temporary(*xData[0]))
  if (numFieldsRead LE 10) then begin
    if (numFieldsRead GT 1) then begin
      callString = 'data = create_struct(temporary(data)'
      for i=1, numFieldsRead-1 do $
        callString = callString + $
          ', strcompress(FieldNames['+STRING(i)+'],/rem)' + $
          ', temporary(*xData['+STRING(i)+'])'
      callString = callString + ')'
      r = EXECUTE(callString)
    endif
  endif else begin
      for i=1, numFieldsRead-1 do $
        data = create_struct(temporary(data), $
           strcompress(FieldNames[i],/rem), temporary(*xData[i]))
  endelse

  ;  Clean up the heap data.
  ;
  for f = 0L, numFieldsRead-1 do $
    PTR_FREE, xData[f]

  ;  Print verbose information.
  ;
  if (doVerbose) then begin
    PRINT, 'Output data ...'
    HELP, data, /STRUCTURES
    PRINT
  endif

  ;  Return the structure.
  ;
  RETURN, data

end         ; read_ascii

; -----------------------------------------------------------------------------

